/******************************************************************************
* (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
*                                                                            *
*  Test of Blk-Memory, all four behavior memories and Data-memory            *
*  Deep-Trace test not yet implemented.                                      *
*  Test is not aborted in case of error -> all occuring errors are written to*
*  report file                                                               *
******************************************************************************/

#include <xselftst.h> 
#include <xaddrmap.h>
#include <xexerdef.h>


/******************************************************************************
*** Data memory
******************************************************************************/

#define BX_DMEM_MEMDEPTH (BestXHasFaust(handle)?(524288UL):131072UL)   /* 4Meg|1Meg = (2 Cols a 4 byte) * 524288|131072 lines */
#define BX_DMEM_MEMWIDTH 2        /* there are 2 cols                        */

/******************************************************************************
* Deep Trace                                                                 *
* Compare with: E2929A#100_IRS.wbk and instr_infile on                       *
* s:\BVSBg\cvt\Projects\PCI-X\HW\DEEPTRACE                                   *
******************************************************************************/

#define BX_DEEPTRACE_OFFSET           0x00070000
#define BX_DEEPTRACE_LATCH_READ_START ((UWORDADDR) (BX_DEEPTRACE_OFFSET + 0x10))
#define BX_DEEPTRACE_LATCH_READ_STOP  ((UWORDADDR) (BX_DEEPTRACE_OFFSET + 0x12))
#define BX_DEEPTRACE_DATA_READ_SDRAM  ((UWORDADDR) (BX_DEEPTRACE_OFFSET + 0x08))
#define BX_DEEPTRACE_ADDR_CTR_REG     ((ULONGADDR) (BX_DEEPTRACE_OFFSET + 0x0C))
#define BX_DEEPTRACE_MEMDEPTH 131072  /*2^21=2Meg= (8 Cols a 2 byte ) * 131072*/
#define BX_DEEPTRACE_MEMWIDTH 8       /* there are 8 cols */

/*******************************************************************************
* further constants used in this file                                         *
******************************************************************************/

#define MASK_COL_MAX 10            /* maximum number columns of mask data */
#define INVALID_MEMOINFOLINE 255   /* defines last possible line */
#define TEST_PATTERN_A 0x55555555  /* Testpattern 1 used for memories */
#define TEST_PATTERN_B 0xaaaaaaaa  /* Testpattern 1 used for memories */

/*******************************************************************************
Typedefs       ***************************************************************
******************************************************************************/

typedef struct
{
  bx_ememtype  typ;               /* type of memory */
  bx_charptrtype  name;         /* name of memory under test */
  bx_int32 NumRows;                /* number of rows */
  bx_int32 NumCols;                /* number of columns */
  bx_int32 ColWidth;              /* width of column in bytes:1,2,4 */
  bx_int32 Min_Reg;                /* Address of Min_Register */
  bx_int32 Max_Reg;                /* Address of Max_Register */
  bx_int32 Ctr;                      /* Address of Counter */
  bx_int32 Data_Reg;              /* Address of DataRegister */
} bx_memoinfotype;

typedef struct
{
  bx_ememtype typ;
  bx_int32  Mask[MASK_COL_MAX]; 
} bx_maskinfotype;







/*******************************************************************************
Prototypes *******************************************************************
******************************************************************************/

static bx_errtype BestXDataMemTest(     bx_handletype handle,
                                   bx_charptrtype file,
                                   bx_int32 start,
                                   bx_int32 rows);
static bx_errtype BestXBehavMemoryTest( bx_handletype handle,
                                       bx_ememtype typ,
                                       bx_charptrtype file,
                                       bx_int32 start,
                                       bx_int32 rows);
static bx_errtype BestXTraceMemTest(    bx_handletype handle,
                                    bx_charptrtype file);
static bx_errtype BestXBlkMemoryTest(   bx_handletype handle,
                                     bx_ememtype typ,
                                     bx_charptrtype file,
                                     bx_int32 start,
                                     bx_int32 rows);



/*******************************************************************************
Initialize     ***************************************************************
******************************************************************************/

/* In the table memoinfo[] all the necessary information about the different  
* memories to be tested is stored. 
* Format is: 
* Memorytyp, MemoryDepth (in Lines), MemoryWidth (in Columns), ColumnSize 
(1,2 or 4 bytes), Min_Register Address, Max_Register Address,Ctr Address, 
Data_Reg Address  
* Compare with: 
* xsetting.h and xaddrmap.h 
* DMEM is not in the table as the testfunction uses BestXDataMemRead() 
* instead of BestXDirectRegBlockWrite() 
*/

static bx_memoinfotype memoinfo[]=
{
  {BX_EMEM_RIBLK,"RIBLK",BX_RIBLK_MEMDEPTH,BX_RIBLK_MEMWIDTH,sizeof(bx_int32),BX_REG_MBLOCK_MIN_REG,BX_REG_MBLOCK_MAX_REG,BX_REG_MBLOCK_CTR,BX_REG_MBLOCK_DATA_REG},
  {BX_EMEM_RIBEH,"RIBEH",BX_RIBEH_MEMDEPTH,BX_RIBEH_MEMWIDTH,sizeof(bx_int16),BX_REG_MBEHAV_MIN_REG,BX_REG_MBEHAV_MAX_REG,BX_REG_MBEHAV_CTR,BX_REG_MBEHAV_DATA_REG},
  {BX_EMEM_CTBEH,"CTBEH",BX_CTBEH_MEMDEPTH,BX_CTBEH_MEMWIDTH,sizeof(bx_int16),BX_REG_TBEHAV_MIN_REG,BX_REG_TBEHAV_MAX_REG,BX_REG_TBEHAV_CTR,BX_REG_TBEHAV_DATA_REG},
  {BX_EMEM_CIBEH,"CIBEH",BX_CIBEH_MEMDEPTH,BX_CIBEH_MEMWIDTH,sizeof(bx_int16),BX_REG_CBEHAV_MIN_REG,BX_REG_CBEHAV_MAX_REG,BX_REG_CBEHAV_CTR,BX_REG_CBEHAV_DATA_REG},
  {BX_EMEM_RTBEH,"RTBEH",BX_RTBEH_MEMDEPTH,BX_RTBEH_MEMWIDTH,sizeof(bx_int16),BX_REG_RTBEHAV_MIN_REG,BX_REG_RTBEHAV_MAX_REG,BX_REG_RTBEHAV_CTR,BX_REG_RTBEHAV_DATA_REG},  
  {BX_EMEM_ENUMSIZE,0,0,0,0,0,0,0}
};



/* In table maskinfo[] the data to be masked out when testing the memory is 
* stored. That is, any valid bit is masked with 1 and an invalid          
* bit is masked with 0. Data in not existing cols is ignored.
* The format is always a 32bit hex-number, independent of the width of the 
* column in the memory (width of column can be 1,2 or 4 bytes).
* If the column-width is smaller than 32bit the mask info is found in the 
* lower 16bit or 8bit respectively 
* Format is: 
* Memorytyp, Mask of column 1, Mask of column 2, .. , Mask of column MASK_COL_MAX 
* Table of unused bits: 
* Memory    m0    m1    m2        m3    m4        m5    m6    m7    m8    m9    
* RIBLK    -    -    -        -    11-31                            
* RIBEH    8-15    7-15    6-15        6-15    7-15        7-15    7-15    7-15    7-15    7-15    
* CTBEH    -    7-15    7-15        7-15    7-15                            
* CIBEH    8-15    7-15    4-5,7-15    7-15    7-15        7-15    7-15    5,7-15    7-15        
* RTBEH    -    7-15    7-15        6-15    0-3,7-15                    
* TMEM   Trace Memory doesn't need an entry as all bits are valid
* Compare with: 
* xmaster.c (RIBLK, RIBEH), xtarget.c (CTBEH), xcomplet.c (CIBEH) and xrequest.x (RTBEH) 
*/
static bx_maskinfotype maskinfo[]=
{
  {BX_EMEM_RIBLK,0xffffffff,0xffffffff,0xffffffff,0xffffffff,0x0000ffff,0x00000000,0x00000000,0x00000000,0x00000000,0x00000000},
  {BX_EMEM_RIBEH,0x000000ff,0x0000007f,0x0000003f,0x0000003f,0x0000007f,0x0000007f,0x0000007f,0x0000007f,0x0000007f,0x0000007f},
  {BX_EMEM_CTBEH,0x0000ffff,0x0000007f,0x0000007f,0x0000007f,0x0000007f,0x00000000,0x00000000,0x00000000,0x00000000,0x00000000},
  {BX_EMEM_CIBEH,0x000000ff,0x0000007f,0x0000004f,0x0000007f,0x0000007f,0x0000007f,0x0000007f,0x0000005f,0x0000007f,0x00000000},
  {BX_EMEM_RTBEH,0x0000ffff,0x0000007f,0x0000007f,0x0000003f,0x00000070,0x00000000,0x00000000,0x00000000,0x00000000,0x00000000},
  {BX_EMEM_ENUMSIZE,0,0,0,0,0,0,0,0,0,0}
};

/*
{BX_EMEM_RIBEH,0x000000ff,0x0000007f,0x0000003f,0x0000003f,0x0000007f,0x0000007f,0x0000007f,0x0000007f,0x0000007f,0x0000007f},
*/  
/*******************************************************************************
Implementation ***************************************************************
******************************************************************************/


/*-----------------------------------------------------------------------------*
* BestXSelfTest()
*
* Purpose: Check Data-,Trace-,Block- and Behavior memories
*-----------------------------------------------------------------------------*/

bx_errtype EXPORT BestXSelfTest(bx_handletype handle,
                                bx_charptrtype file,
                                bx_int32 rows,
                                bx_int32 start,
                                bx_selftsttype dev) 
{
  BX_DECLARE_FUNCNAME("BestXSelfTest [selftest]");

  BX_TRY_VARS;

  FILE *fp=NULL;       /* used to store resulting infos in file */
  char *info=NULL;

  BX_TRY_BEGIN
  {

    fp=BESTX_FOPEN(file,"w"); 

    if (fp==NULL)
    {
      return BX_E_FILE_OPEN;
    }

    BX_TRY(BestXVersionRead(handle,BX_VERSION_CAPI,&info));

    /* REMOVE: only for debug
    BESTX_FPRINTF(stdout,
    "Writing test results in report file %s\nThis may take a few minutes...\n",
    file);
    */
    BESTX_FPRINTF(fp,"Selftest report file\n");
    BESTX_FPRINTF(fp,"CAPI version: %s\n",info);

    if (fp!=NULL)
    {
      BESTX_FCLOSE(fp);
    }

    switch(dev)
    {
    case BX_SELFTST_RIBLK:
      __status|=BestXBlkMemoryTest(handle,BX_EMEM_RIBLK,file,start,rows); 
      break;

    case BX_SELFTST_RIBEH:
      __status|=BestXBehavMemoryTest(handle,BX_EMEM_RIBEH,file,start,rows); 
      break;

    case BX_SELFTST_CTBEH:
      __status|=BestXBehavMemoryTest(handle,BX_EMEM_CTBEH,file,start,rows); 
      break;

    case BX_SELFTST_CIBEH:
      __status|=BestXBehavMemoryTest(handle,BX_EMEM_CIBEH,file,start,rows); 
      break;

    case BX_SELFTST_RTBEH:
      __status|=BestXBehavMemoryTest(handle,BX_EMEM_RTBEH,file,start,rows); 
      break;

    case BX_SELFTST_TRACE:
      __status|=BestXTraceMemTest(handle,file);
      break;

    case BX_SELFTST_DATA:
      __status|=BestXDataMemTest(handle,file,start,rows);
      break;

    case BX_SELFTST_ALL:
      __status|=BestXDataMemTest(handle,file,start,rows);    
      __status|=BestXBlkMemoryTest(handle,BX_EMEM_RIBLK,file,start,rows);
      __status|=BestXBehavMemoryTest(handle,BX_EMEM_RIBEH,file,start,rows); 
      __status|=BestXBehavMemoryTest(handle,BX_EMEM_CTBEH,file,start,rows); 
      __status|=BestXBehavMemoryTest(handle,BX_EMEM_CIBEH,file,start,rows); 
      __status|=BestXBehavMemoryTest(handle,BX_EMEM_RTBEH,file,start,rows); 
      __status|=BestXTraceMemTest(handle,file);     
      break;

    default:
      BX_TRY(BX_E_INVALID_CASE);

    } /*switch(dev)*/

    BX_TRY(__status);
  }

  BX_ERRETURN(BX_TRY_RET);
}





/*******************************************************************************
* Test of the 4 Behaviormemories                                              *
******************************************************************************/


static bx_errtype BestXBehavMemoryTest(
                                       bx_handletype handle,
                                       bx_ememtype typ,
                                       bx_charptrtype file,
                                       bx_int32 start,
                                       bx_int32 rows
                                       )
{
  BX_DECLARE_FUNCNAME("BestXBehavMemoryTest [behavmemorytest]");

  BX_TRY_VARS;

  bx_int32 NumBytes;            /* Memory Space needed                    */
  bx_int32 Row,Col;             /* needed for for(..) loops               */
  bx_int16ptr SData = NULL;     /* Pointer on Data to be stored           */
  bx_int16ptr RData = NULL;     /* Pointer on Data to be read             */
  bx_int16ptr SData_act;        /* Pointer to actual value in memoryblock */
  bx_int16ptr RData_act;        /* */
  bx_int32 i;                   /* used in for() loop                     */
  bx_int32 index_memoinfo;      /* actual index in memoinfo               */
  bx_int32 index_maskinfo;      /* actual index in maskinfo               */
  bx_int16 pattern[2];          /* Test-Pattern  is stored in here        */
  bx_int32 ecount;              /* number of errors when reading memory   */
  bx_int32 ecount_all=0;        /* is memory ok */
  bx_int8 Mem_Read_Error;       /* used to indicate memory read error      */
  bx_int8  index_pattern;       /* to select pattern a or b */
  bx_int32 StartOffset;         /* Indicates Offset for DirectRegs */
  FILE *fp=NULL;            /* used for storing resulting infos in file */



  BX_TRY_BEGIN {

    /* open report file */
    fp=BESTX_FOPEN(file,"a");

    if (fp==NULL)
    {
      return BX_E_FILE_OPEN;
    }

    /* checks if required info is stored in tables memoinfo and maskinfo */
    /* by walking through both tables                 */

    index_memoinfo=INVALID_MEMOINFOLINE; 
    i=0;
    do   /*  */  
    {
      if (memoinfo[i].typ==typ)
      {
        index_memoinfo=i;     
      }
    }  while (memoinfo[++i].typ!=BX_EMEM_ENUMSIZE);


    index_maskinfo=INVALID_MEMOINFOLINE;          
    i=0;
    do   /*  */  
    {
      if (maskinfo[i].typ==typ)
      {
        index_maskinfo=i;     
      }
    } while (maskinfo[++i].typ!=BX_EMEM_ENUMSIZE);


    /* if required infos are found, go  on ... */
    if((index_maskinfo!=INVALID_MEMOINFOLINE) 
      && (index_memoinfo!=INVALID_MEMOINFOLINE))
    {

      /* load testpattern into array */
      pattern[0]=(bx_int16)TEST_PATTERN_A; 
      pattern[1]=(bx_int16)TEST_PATTERN_B; 

      /* (check range) set number of rows to be tested */
      if(rows>memoinfo[index_memoinfo].NumRows) 
        rows=memoinfo[index_memoinfo].NumRows;
      if(rows<=0) rows=1;
      if(start>BX_DMEM_MEMDEPTH) start=BX_DMEM_MEMDEPTH;
      if( (start+rows)>memoinfo[index_memoinfo].NumRows)
        rows=memoinfo[index_memoinfo].NumRows-start;
      NumBytes=rows*memoinfo[index_memoinfo].NumCols
        *memoinfo[index_memoinfo].ColWidth;
      StartOffset=start*memoinfo[index_memoinfo].NumCols
        *memoinfo[index_memoinfo].ColWidth;

      /* information text */
      BESTX_FPRINTF(  fp,
        "\n********************************************************************************\n");
      BESTX_FPRINTF(  fp,
        "Testing %i rows on memory %s, starting on row %i:\n",
        rows,
        memoinfo[index_memoinfo].name,
        start
        );
      BESTX_FPRINTF(  fp,"\nAdditional info:\n");
      BESTX_FPRINTF(  fp,  "----------------\n");

      BESTX_FPRINTF(fp,"This memory has %i rows and %i columns. Each column is %i bytes wide.\n\n",
        memoinfo[index_memoinfo].NumRows,
        memoinfo[index_memoinfo].NumCols,
        memoinfo[index_memoinfo].ColWidth
        );

      BESTX_FPRINTF(fp,"\nIn order to test the memory, two testpattern A and B are written\n");
      BESTX_FPRINTF(fp,"to the card memory, read and compared.\n"); 
      BESTX_FPRINTF(fp,"As not every bit can be used a mask is for each column is\n");
      BESTX_FPRINTF(fp,"created.\n");
      BESTX_FPRINTF(fp,"Mask&PA is mask and testpattern A (bitwise AND) and Mask&PB is\n");
      BESTX_FPRINTF(fp,"mask and testpattern B.\n");



      BESTX_FPRINTF(fp,"\nColumn :");
      for(i=0;i<memoinfo[index_memoinfo].NumCols;i++) 
        BESTX_FPRINTF(fp,"%.2i    ",i);
      BESTX_FPRINTF(fp,"\nMask   :");
      for(i=0;i<memoinfo[index_memoinfo].NumCols;i++) 
        BESTX_FPRINTF(fp,"x%.4x ",(bx_int16)maskinfo[index_maskinfo].Mask[i]);
      BESTX_FPRINTF(fp,"\nMask&PA:");
      for(i=0;i<memoinfo[index_memoinfo].NumCols;i++) 
        BESTX_FPRINTF(fp,"x%.4x ",pattern[0] & maskinfo[index_maskinfo].Mask[i]);
      BESTX_FPRINTF(fp,"\nMask&PB:");
      for(i=0;i<memoinfo[index_memoinfo].NumCols;i++) 
        BESTX_FPRINTF(fp,"x%.4x ",pattern[1] & maskinfo[index_maskinfo].Mask[i]);
      BESTX_FPRINTF(fp,"\n");


      BESTX_FPRINTF(  fp,"\nNow starting test ...\n");
      BESTX_FPRINTF(  fp,"---------------------\n"
        );

      /* initializing memory to store */
      SData = (bx_int16ptr) BestXMemMalloc((size_t)NumBytes); 
      if (SData == (bx_int16ptr) 0)
      {
        BX_TRY_ERROR(BX_E_HOST_MEM_FULL);
      }

      /* initializying memory to read */
      RData = (bx_int16ptr) BestXMemMalloc((size_t)NumBytes);
      if (RData == (bx_int16ptr) 0)
      {
        BX_TRY_ERROR(BX_E_HOST_MEM_FULL);
      }

      /* Switch to Prog-Mode (DBI clock) */
      BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

      /* now both test pattern pattern[2] will be written, read and compared */
      for(index_pattern=0;index_pattern<2;index_pattern++)
      {
        /* Initit card Min_Reg and Max_Reg (defines start and stop column) */
        BX_TRY(BestXDirectRegWrite( handle,
          memoinfo[index_memoinfo].Min_Reg,
          sizeof(bx_int16),
          0
          ));

        BX_TRY(BestXDirectRegWrite( handle,
          memoinfo[index_memoinfo].Max_Reg,
          sizeof(bx_int16),
          memoinfo[index_memoinfo].NumCols-1
          ));
        /* initialize pointer to begin of allocated memoryblock */
        SData_act=SData;

        /* Writes masked data in memory on host */
        for(Row=0;Row<rows;Row++) /* memoinfo[index_memoinfo].NumRows;Row++) */
        {
          for(Col=0;Col<memoinfo[index_memoinfo].NumCols;Col++)
          {

            SData_act[Row*memoinfo[index_memoinfo].NumCols+Col]
            =(bx_int16)( pattern[index_pattern] 
            & maskinfo[index_maskinfo].Mask[Col] );

          } /* for(Col..) */
        } /* for(Row..) */


        /* Set starting row (i.e. offset) from which to start programming */
        BX_TRY(BestXDirectRegWrite( handle,
          memoinfo[index_memoinfo].Ctr,
          sizeof(bx_int16),
          StartOffset
          ));


        /* now store memory on card */     
        BX_TRY(BestXDirectRegBlockWrite(  handle,
          memoinfo[index_memoinfo].Data_Reg,    
          memoinfo[index_memoinfo].ColWidth,/*width of HW-Register */            
          1,                 /*autoinc */
          (bx_int8ptr)SData, /*first byte */
          sizeof(bx_int16),  /*size of pointer in SW */
          NumBytes
          ));                

        /* set starting row from which to start reading */
        BX_TRY(BestXDirectRegWrite( handle,
          memoinfo[index_memoinfo].Ctr,
          sizeof(bx_int16),
          StartOffset
          ));

        /* now read the data back to host*/
        BX_TRY(BestXDirectRegBlockRead(  handle,
          memoinfo[index_memoinfo].Data_Reg,    
          memoinfo[index_memoinfo].ColWidth, /*width of HW-Register */            
          1,                 /*autoinc */
          (bx_int8ptr)RData, /*first byte */
          sizeof(bx_int16),  /*size of pointer in SW */
          NumBytes
          ));        


        /* now Parse Memory for errors */
        BESTX_FPRINTF(fp,"\noffset\t\trow\t\tcol\t\twritten\t\tread (masked)\n");
        RData_act=RData; /* set pointer to begin of memory block*/
        ecount=0;        /* count occuring errors */

        for(Row=0;Row<rows;Row++) 
        {
          for(Col=0;Col<memoinfo[index_memoinfo].NumCols;Col++)
          {
            Mem_Read_Error=0;

            if((SData[Row*memoinfo[index_memoinfo].NumCols+Col])
              !=(RData[Row*memoinfo[index_memoinfo].NumCols+Col]
              & maskinfo[index_maskinfo].Mask[Col])) 
              {
                Mem_Read_Error=1;
                ecount++;
                ecount_all=1;
              }

              if(Mem_Read_Error)
              {
                /* write result in table: address,written pattern, read pattern*/
                BESTX_FPRINTF(  fp,
                  "0x%.4x\t\t%i\t\t%i\t\t0x%.4x\t\t0x%.4x\n",
                  memoinfo[index_memoinfo].ColWidth
                  *((Row+start)*memoinfo[index_memoinfo].NumCols+Col),
                  (Row+start),
                  Col,
                  (SData[Row*memoinfo[index_memoinfo].NumCols+Col]),
                  (RData[Row*memoinfo[index_memoinfo].NumCols+Col] 
                  & maskinfo[index_maskinfo].Mask[Col])
                    );
                  Mem_Read_Error=0;
              }

          } /* close for(Col=0 ... */ 
        } /*for (Row...) */

        BESTX_FPRINTF(fp,"\n%i errors occured when writing pattern 0x%.4x on memory %s.\n\n",
          ecount,
          pattern[index_pattern],
          memoinfo[index_memoinfo].name
          );  

      } /* end of for(i=0;i<2;i++) (to write both testpattern) */

      /*free memory */  
      BestXMemFree((void**)&SData);
      BestXMemFree((void**)&RData);

      /* Switch back to Run-Mode (PCI clock) */  
      BX_TRY_PROGRESS(BX_E_OK);
      BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));

      /* REMOVE: just for debug ... */
      /* result to screen 
      if(!ecount_all)
      {
      BESTX_FPRINTF(stdout,"Memory %s is ok...\n",memoinfo[index_memoinfo].name);  
      }
      else
      {
      BESTX_FPRINTF(stdout,"Memory %s is not ok !. Details see report file\n",memoinfo[index_memoinfo].name);
      }
      */
    } /* end of if(act_ind_mask!=255 ... */

    BESTX_FPRINTF(  fp,
      "End of test on memory %s:\n",
      memoinfo[index_memoinfo].name
      );
    BESTX_FPRINTF(  fp,
      "\n********************************************************************************\n");


    /* close file */

    if (fp!=NULL)
    {
      BESTX_FCLOSE(fp);
    }

  } /* end of BX_TRY_BEGIN */

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        if (SData)
        {
          BestXMemFree((void**)&SData);
        }
        if (RData)
        {
          BestXMemFree((void**)&RData);
        }
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }

  if(BX_TRY_RET)
  {
    /* error occurred */
    BX_ERRETURN(BX_TRY_RET);
  }
  else
  {
    /* no error occurred */
    if(!ecount_all) 
    {
      BX_ERRETURN(BX_E_OK);
    }
    else
    {
      BX_ERRETURN(BX_E_SELFTEST_FAILED);
    }
  }
}







/*******************************************************************************
* Test of the Blockmemory                                                     *
******************************************************************************/


static bx_errtype BestXBlkMemoryTest(
                                     bx_handletype handle,
                                     bx_ememtype typ,
                                     bx_charptrtype file,
                                     bx_int32 start,
                                     bx_int32 rows
                                     )
{
  BX_DECLARE_FUNCNAME("BestXBlkMemoryTest [blkmemorytest]");

  BX_TRY_VARS;

  bx_int32 NumBytes;            /* Memory Space needed                    */
  bx_int32 Row,Col;             /* needed for for(..) loops               */
  bx_int32ptr SData = NULL;     /* Pointer on Data to be stored           */
  bx_int32ptr RData = NULL;     /* Pointer on Data to be read             */
  bx_int32ptr SData_act;        /* Pointer to actual value in memoryblock */
  bx_int32ptr RData_act;        /* */
  bx_int32 i;                   /* used in for() loop                     */
  bx_int32 index_memoinfo;      /* actual index in memoinfo               */
  bx_int32 index_maskinfo;      /* actual index in maskinfo               */
  bx_int32 pattern[2];          /* Test-Pattern  is stored in here        */
  bx_int32 ecount;              /* number of errors when reading memory   */
  bx_int32 ecount_all=0;        /* test failed or not ... */
  bx_int8 Mem_Read_Error;       /* used to indicate memory read error     */
  bx_int8  index_pattern;       /* to select pattern a or b               */
  bx_int32 StartOffset;            /* Offset from which to start DirectRegWr */
  FILE *fp=NULL;        /* used for storing resulting infos in file */



  BX_TRY_BEGIN  {

    /* open report file to append new info */
    fp=BESTX_FOPEN(file,"a");
    /* fp=stdout;  */

    if (fp==NULL)
    {
      return BX_E_FILE_OPEN;
    }

    /* checks if required info is stored in tables memoinfo and maskinfo */
    /* by walking through both tables                 */

    index_memoinfo=INVALID_MEMOINFOLINE; 
    i=0;
    do   
    {
      if (memoinfo[i].typ==typ)
      {
        index_memoinfo=i;     
      }
    }  while (memoinfo[++i].typ!=BX_EMEM_ENUMSIZE);


    index_maskinfo=INVALID_MEMOINFOLINE;          
    i=0;
    do   /*  */  
    {
      if (maskinfo[i].typ==typ)
      {
        index_maskinfo=i;     
      }
    } while (maskinfo[++i].typ!=BX_EMEM_ENUMSIZE);




    /* if required infos are found, go  on ... */
    if((index_maskinfo!=INVALID_MEMOINFOLINE) 
      && (index_memoinfo!=INVALID_MEMOINFOLINE))
    {

      /* load testpattern into array */
      pattern[0]=TEST_PATTERN_A; 
      pattern[1]=TEST_PATTERN_B; 

      /* (check range) set number of rows to be tested */
      if(rows>memoinfo[index_memoinfo].NumRows) 
        rows=memoinfo[index_memoinfo].NumRows;
      if(rows<=0) rows=1;
      if(start>BX_DMEM_MEMDEPTH) start=BX_DMEM_MEMDEPTH;
      if( (start+rows)>memoinfo[index_memoinfo].NumRows)
        rows=memoinfo[index_memoinfo].NumRows-start;
      NumBytes=rows*memoinfo[index_memoinfo].NumCols
        *memoinfo[index_memoinfo].ColWidth;
      StartOffset=start*memoinfo[index_memoinfo].NumCols
        *memoinfo[index_memoinfo].ColWidth;

      /* information text */
      BESTX_FPRINTF(  fp,
        "\n********************************************************************************\n");
      BESTX_FPRINTF(  fp,
        "Testing %i rows on memory %s, starting on row %i:\n",
        rows,
        memoinfo[index_memoinfo].name,
        start
        );
      BESTX_FPRINTF(  fp,"\nAdditional info:\n");
      BESTX_FPRINTF(  fp,  "----------------\n");

      BESTX_FPRINTF(fp,"This memory has %i rows and %i columns. Each column is %i bytes wide.\n\n",
        memoinfo[index_memoinfo].NumRows,
        memoinfo[index_memoinfo].NumCols,
        memoinfo[index_memoinfo].ColWidth
        );

      BESTX_FPRINTF(fp,"\nIn order to test the memory, two testpattern A and B are written\n");
      BESTX_FPRINTF(fp,"to the card memory, read and compared.\n"); 
      BESTX_FPRINTF(fp,"As not every bit can be used a mask is for each column is\n");
      BESTX_FPRINTF(fp,"created.\n");
      BESTX_FPRINTF(fp,"Mask&PA is mask and testpattern A (bitwise AND) and Mask&PB is\n");
      BESTX_FPRINTF(fp,"mask and testpattern B.\n");

      BESTX_FPRINTF(fp,"\nColumn :");
      for(i=0;i<memoinfo[index_memoinfo].NumCols;i++) BESTX_FPRINTF(fp,"%.2i        ",i);
      BESTX_FPRINTF(fp,"\nMask   :");
      for(i=0;i<memoinfo[index_memoinfo].NumCols;i++) 
        BESTX_FPRINTF(fp,"x%.8x ",maskinfo[index_maskinfo].Mask[i]);
      BESTX_FPRINTF(fp,"\nMask&PA:");
      for(i=0;i<memoinfo[index_memoinfo].NumCols;i++) 
        BESTX_FPRINTF(fp,"x%.8x ",pattern[0] & maskinfo[index_maskinfo].Mask[i]);
      BESTX_FPRINTF(fp,"\nMask&PB:");
      for(i=0;i<memoinfo[index_memoinfo].NumCols;i++) 
        BESTX_FPRINTF(fp,"x%.8x ",pattern[1] & maskinfo[index_maskinfo].Mask[i]);
      BESTX_FPRINTF(fp,"\n");

      BESTX_FPRINTF(  fp,"\nNow starting test ...\n");
      BESTX_FPRINTF(  fp,"---------------------\n"
        );

      /* memory to store */
      SData = (bx_int32ptr) BestXMemMalloc((size_t)NumBytes); 
      if (SData == (bx_int32ptr) 0)
      {
        BX_TRY_ERROR(BX_E_HOST_MEM_FULL);
      }

      /* memory to read */
      RData = (bx_int32ptr) BestXMemMalloc((size_t)NumBytes);
      if (RData == (bx_int32ptr) 0)
      {
        BX_TRY_ERROR(BX_E_HOST_MEM_FULL);
      }

      /* Switch to Prog-Mode (DBI clock) */
      BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

      /* now both test pattern pattern[2] will be written, read and compared */
      for(index_pattern=0;index_pattern<2;index_pattern++)
      {



        /* Initit card Min_Reg and Max_Reg (defines start and stop column) */
        BX_TRY(BestXDirectRegWrite( handle,
          memoinfo[index_memoinfo].Min_Reg,
          sizeof(bx_int16),
          0
          ));

        BX_TRY(BestXDirectRegWrite( handle,
          memoinfo[index_memoinfo].Max_Reg,
          sizeof(bx_int16),
          memoinfo[index_memoinfo].NumCols-1
          ));

        /* init all memory with 0 for test */
        SData_act=SData;
        RData_act=RData;

        for(Row=0;Row<rows;Row++) 
        {
          for(Col=0;Col<BX_DMEM_MEMWIDTH;Col++)
          {
            SData_act[Row*BX_DMEM_MEMWIDTH+Col]=0;
            RData_act[Row*BX_DMEM_MEMWIDTH+Col]=0;
          } 
        } 



        /* initialize pointer to begin of allocated memoryblock */
        SData_act=SData;

        /* Writes masked data in memory on host */
        for(Row=0;Row<rows;Row++) /* memoinfo[index_memoinfo].NumRows;Row++) */
        {
          for(Col=0;Col<memoinfo[index_memoinfo].NumCols;Col++)
          {

            SData_act[Row*memoinfo[index_memoinfo].NumCols+Col]
            =(pattern[index_pattern] & maskinfo[index_maskinfo].Mask[Col] );
          } 
        } 


        /* Set starting row (i.e. offset) from which to start programming */
        BX_TRY(BestXDirectRegWrite( handle,
          memoinfo[index_memoinfo].Ctr,
          sizeof(bx_int16),
          StartOffset
          ));


        /* now store memory on card */     
        /* BESTX_FPRINTF(fp,"Write %i Bytes of 0x%x\n",NumBytes,pattern[i]); */ 
        BX_TRY(BestXDirectRegBlockWrite(  handle,
          memoinfo[index_memoinfo].Data_Reg,    
          memoinfo[index_memoinfo].ColWidth,  /*width of HW-Register */            
          1,                 /*autoinc */
          (bx_int8ptr)SData, /*first byte */
          sizeof(bx_int32),  /*size of pointer in SW */
          NumBytes
          ));                

        /* Set starting row from which to start reading memory */
        BX_TRY(BestXDirectRegWrite( handle,
          memoinfo[index_memoinfo].Ctr,
          sizeof(bx_int16),
          StartOffset
          ));

        /* read the data back to host*/
        BX_TRY(BestXDirectRegBlockRead(  handle,
          memoinfo[index_memoinfo].Data_Reg,    
          memoinfo[index_memoinfo].ColWidth, /*width of HW-Register */            
          1,                 /*autoinc */
          (bx_int8ptr)RData, /*first byte */
          sizeof(bx_int32),  /*size of pointer in SW */
          NumBytes
          ));        


        /* Parse Memory for errors */

        BESTX_FPRINTF(fp,"\noffset\t\trow\tcol\twritten\t\tread (masked)\n");
        RData_act=RData; /* set pointer to begin of memory block*/
        SData_act=SData; /* set pointer to begin of 'stored memory' */
        ecount=0;   /* count occuring errors */

        for(Row=0;Row<rows;Row++) /* debug...memoinfo[index_memoinfo].NumRows;Row++) */
        {
          for(Col=0;Col<memoinfo[index_memoinfo].NumCols;Col++)
          {
            Mem_Read_Error=0;

            if((SData[Row*memoinfo[index_memoinfo].NumCols+Col])
              !=(RData[Row*memoinfo[index_memoinfo].NumCols+Col]
              & maskinfo[index_maskinfo].Mask[Col]))
              {
                Mem_Read_Error=1;
                ecount++;
                ecount_all=1;
              }

              if(Mem_Read_Error)
              {
                /* write result in table: address,written pattern, read pattern*/
                BESTX_FPRINTF(fp,
                  "0x%8.8x\t%i\t%i\t0x%.8x\t0x%.8x\n",
                  memoinfo[index_memoinfo].ColWidth
                  *((Row+start)*memoinfo[index_memoinfo].NumCols+Col),
                  (Row+start),
                  Col,
                  (SData_act[Row*memoinfo[index_memoinfo].NumCols+Col]),
                  (RData_act[Row*memoinfo[index_memoinfo].NumCols+Col] 
                  & maskinfo[index_maskinfo].Mask[Col])
                    );

                  Mem_Read_Error=0;
              }

          } /* close for(Col=0 ... */ 
        } /*for (Row...) */

        BESTX_FPRINTF(fp,"\n%i errors occured when writing pattern 0x%.8x on memory %s.\n\n",
          ecount,
          pattern[index_pattern],
          memoinfo[index_memoinfo].name
          );
      } /* end of for(index_pattern=0;index_pattern<2;index_pattern++) (to write both testpattern) */

      /*free memory */  
      BestXMemFree((void**)&SData);
      BestXMemFree((void**)&RData);

      /* Switch back to Run-Mode (PCI clock) */
      BX_TRY_PROGRESS(BX_E_OK);
      BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));


      /* REMOVE: only for debug .. */
      /* result to screen 
      if(!ecount_all)
      {
      BESTX_FPRINTF(stdout,"Memory %s is ok...\n",memoinfo[index_memoinfo].name);  
      }
      else
      {
      BESTX_FPRINTF(stdout,"Memory %s is not ok !. Details see report file ...\n",memoinfo[index_memoinfo].name);
      }
      */


    } /* end of if(act_ind_mask!=255 ... */

    BESTX_FPRINTF(  fp,
      "End of test on memory %s:\n",
      memoinfo[index_memoinfo].name
      );
    BESTX_FPRINTF(  fp,
      "\n********************************************************************************\n");


    /* close file */

    if (fp!=NULL)
    {
      BESTX_FCLOSE(fp);
    }

  } /* end of BX_TRY_BEGIN */

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        if (SData)
        {
          BestXMemFree((void**)&SData);
        }
        if (RData)
        {
          BestXMemFree((void**)&RData);
        }
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }

  if(BX_TRY_RET) BX_ERRETURN(BX_TRY_RET);
  if(!ecount_all) BX_ERRETURN(BX_E_OK); else BX_ERRETURN(BX_E_SELFTEST_FAILED);

}



/*******************************************************************************
* Test of Datamemory                                                          * 
*******************************************************************************
* #define BX_DMEM_MEMDEPTH 131072 or 524288    1Meg/4Meg = (2 Cols a 4 byte) * 131072/524288 lines
* #define BX_DMEM_MEMWIDTH 2        there are 2 cols 
*/

static bx_errtype BestXDataMemTest(
                                   bx_handletype handle,
                                   bx_charptrtype file,
                                   bx_int32 start,
                                   bx_int32 rows
                                   )
{
  BX_DECLARE_FUNCNAME("BestXDataMemTest [datamemtest]");

  BX_TRY_VARS;

  bx_int8  i;                   /* index for address line test                   */
  bx_int8  memdeep;             /* DataMem Adress Lines */
  bx_int32 Pattern;             /* data pattern that is written to data memory   */
  bx_int32 MemAddr;             /* used for addressing data memory               */
  bx_int32 ReadValHi,ReadValLo; /* used for read data from memory                */
  bx_bool  TestFail = BX_FALSE; /* error occured flag                            */
  FILE *fp=NULL;        /* used for storing resulting infos in file      */



  BX_TRY_BEGIN
  {

    fp=BESTX_FOPEN(file,"a");

    if (fp==NULL)
    {
      return BX_E_FILE_OPEN;
    }


    /* information text */
    BESTX_FPRINTF(  fp,
      "\n************************************************************************\n");
    BESTX_FPRINTF(  fp,"Testing Data-memory\n");

    BESTX_FPRINTF(  fp,"Now starting test ...\n");
    BESTX_FPRINTF(  fp,"---------------------\n");


    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));


    /* Init card Min_Reg and Max_Reg (defines start and stop column) */
    BX_TRY(BestXDirectRegWrite( handle,
      BX_REG_DMEM_MIN_REG,
      sizeof(bx_int16),
      0
      ));

    BX_TRY(BestXDirectRegWrite( handle,
      BX_REG_DMEM_MAX_REG,
      sizeof(bx_int16),
      BX_DMEM_MEMWIDTH-1
      ));

    /* First test all data lines */
    /* Write different data pattern into memory line 0*/

    MemAddr = 0x0;

    /* Set starting row (i.e. offset) from which to start writing */
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_ADDR_CTR,sizeof(bx_int32),MemAddr));
    Pattern = 0x0;

    /* write Pattern to memory line 0 */
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),Pattern));
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),Pattern));


    /* and compare */
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_ADDR_CTR,sizeof(bx_int32),MemAddr));
    BX_TRY(BestXDirectRegRead( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),&ReadValLo));
    BX_TRY(BestXDirectRegRead( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),&ReadValHi));

    if ((ReadValLo != Pattern) || (ReadValHi != Pattern))
    {
      TestFail = BX_TRUE;
      BESTX_FPRINTF(  fp,"Error in data memory data line test.\n");
      BESTX_FPRINTF(  fp,"Wrote Value %08X %08X and read back %08X %08X\n", 
        Pattern, Pattern, ReadValHi, ReadValLo);
    }

    /* now try a different Pattern */

    /* Set starting row (i.e. offset) from which to start writing */
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_ADDR_CTR,sizeof(bx_int32),MemAddr));
    Pattern = 0xFFFFFFFF;

    /* write Pattern to memory line 0 */
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),Pattern));
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),Pattern));


    /* and compare */
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_ADDR_CTR,sizeof(bx_int32),MemAddr));
    BX_TRY(BestXDirectRegRead( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),&ReadValLo));
    BX_TRY(BestXDirectRegRead( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),&ReadValHi));

    if ((ReadValLo != Pattern) || (ReadValHi != Pattern))
    {
      TestFail = BX_TRUE;
      BESTX_FPRINTF(  fp,"Error in data memory data line test.\n");
      BESTX_FPRINTF(  fp,"Wrote Value %08X %08X and read back %08X %08X\n", 
        Pattern, Pattern, ReadValHi, ReadValLo);
    }

    /* now try a different Pattern */

    /* Set starting row (i.e. offset) from which to start writing */
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_ADDR_CTR,sizeof(bx_int32),MemAddr));
    Pattern = 0xAAAAAAAA;

    /* write Pattern to memory line 0 */
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),Pattern));
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),Pattern));


    /* and compare */
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_ADDR_CTR,sizeof(bx_int32),MemAddr));
    BX_TRY(BestXDirectRegRead( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),&ReadValLo));
    BX_TRY(BestXDirectRegRead( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),&ReadValHi));

    if ((ReadValLo != Pattern) || (ReadValHi != Pattern))
    {
      TestFail = BX_TRUE;
      BESTX_FPRINTF(  fp,"Error in data memory data line test.\n");
      BESTX_FPRINTF(  fp,"Wrote Value %08X %08X and read back %08X %08X\n", 
        Pattern, Pattern, ReadValHi, ReadValLo);
    }

    /* now try a different Pattern */

    /* Set starting row (i.e. offset) from which to start writing */
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_ADDR_CTR,sizeof(bx_int32),MemAddr));
    Pattern = 0x55555555;

    /* write Pattern to memory line 0 */
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),Pattern));
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),Pattern));


    /* and compare */
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_ADDR_CTR,sizeof(bx_int32),MemAddr));
    BX_TRY(BestXDirectRegRead( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),&ReadValLo));
    BX_TRY(BestXDirectRegRead( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),&ReadValHi));

    if ((ReadValLo != Pattern) || (ReadValHi != Pattern))
    {
      TestFail = BX_TRUE;
      BESTX_FPRINTF(  fp,"Error in data memory data line test.\n");
      BESTX_FPRINTF(  fp,"Wrote Value %08X %08X and read back %08X %08X\n", 
        Pattern, Pattern, ReadValHi, ReadValLo);
    }

    /* Now test address lines by writing different pattern to byte addresses 0,8,16 ... 128K */
    /* check if E2929A or E2930A new RaM 26.02. */
    if (BestXHasFaust(handle)== 1) /* E2930A */
    { 
      memdeep = 19;
    }
    else /* E2929B */
    {
      memdeep = 17;
    }

    MemAddr = 0x0;
    Pattern = 0x0;

    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_ADDR_CTR,sizeof(bx_int32),MemAddr));
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),Pattern));
    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),Pattern));

    /* as we have QWORD addresses start with addr(3) */ 
    MemAddr = 0x8;
    Pattern = 0x8;
    for (i=0 ; i<memdeep ; i++)
    {
      BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_ADDR_CTR,sizeof(bx_int32),MemAddr));
      BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),Pattern));
      BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),Pattern));

      MemAddr <<= 1;
      Pattern <<= 1;
    }

    /* now compare data */
    MemAddr = 0x0;
    Pattern = 0x0;

    BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_ADDR_CTR,sizeof(bx_int32),MemAddr));
    BX_TRY(BestXDirectRegRead( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),&ReadValLo));
    BX_TRY(BestXDirectRegRead( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),&ReadValHi));

    if ((ReadValLo != Pattern) || (ReadValHi != Pattern))
    {
      TestFail = BX_TRUE;
      BESTX_FPRINTF(  fp,"Error in data memory address line test.\n");
      BESTX_FPRINTF(  fp,"Wrote Value %08X %08X to Address %08X and read back %08X %08X\n", 
        Pattern, Pattern, MemAddr, ReadValHi, ReadValLo);
    }


    /* as we have QWORD addresses start with addr(3) */ 
    MemAddr = 0x8;
    Pattern = 0x8;
    for (i=0 ; i<memdeep ; i++)
    {
      BX_TRY(BestXDirectRegWrite( handle,BX_REG_DMEM_ADDR_CTR,sizeof(bx_int32),MemAddr));
      BX_TRY(BestXDirectRegRead( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),&ReadValLo));
      BX_TRY(BestXDirectRegRead( handle,BX_REG_DMEM_DATA_REG,sizeof(bx_int32),&ReadValHi));
      if ((ReadValLo != Pattern) || (ReadValHi != Pattern))
      {
        TestFail = BX_TRUE;
        BESTX_FPRINTF(  fp,"Error in data memory address line test.\n");
        BESTX_FPRINTF(  fp,"Wrote Value %08X %08X to Address %08X and read back %08X %08X\n", 
          Pattern, Pattern, MemAddr, ReadValHi, ReadValLo);
      }

      MemAddr <<= 1;
      Pattern <<= 1;
    }





    /* Switch back to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));

    BESTX_FPRINTF(  fp,"End of test on Data-memory:\n");
    BESTX_FPRINTF(  fp,"\n************************************************************************\n");


    /* close file */

    if (fp!=NULL)
    {
      BESTX_FCLOSE(fp);
    }

  } /* end of BX_TRY_BEGIN */

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }

  if(BX_TRY_RET) BX_ERRETURN(BX_TRY_RET);
  if(!TestFail) BX_ERRETURN(BX_E_OK); else BX_ERRETURN(BX_E_SELFTEST_FAILED);

} /* end of datamemtest */




/*******************************************************************************
* DeepTrace Memory Test                                                       * 
* (cannot be written via DBI yet, therefore no test possible)                 *
******************************************************************************/
/* {TMEM,"TMEM",BX_DEEPTRACE_MEMDEPTH,BX_DEEPTRACE_MEMWIDTH,sizeof(bx_int16),BX_DEEPTRACE_LATCH_READ_START,BX_DEEPTRACE_LATCH_READ_STOP,BX_DEEPTRACE_ADDR_CTR_REG,BX_DEEPTRACE_DATA_READ_SDRAM},
*/

static bx_errtype BestXTraceMemTest(
                                    bx_handletype handle,
                                    bx_charptrtype file
                                    )
{

  BX_DECLARE_FUNCNAME("BestXTraceMemTest [tracememtst]");
  BX_TRY_VARS_NO_PROG;

  bx_int8  i,j;                 /* index for address line test                   */
  bx_int32 Pattern;             /* data pattern that is written to data memory   */
  bx_int32 MemAddr;             /* used for addressing data memory               */
  bx_int32 ReadVal;             /* used for read data from memory                */
  bx_int32 ReadValLo;
  bx_int32 ReadValHi;
  bx_bool  TestFail = BX_FALSE; /* error occured flag                            */
  bx_int8 faust=1, memsize=0, memdeep=0;
  bx_int32  maxreg=0;

  FILE *fp=NULL;

  BX_TRY_BEGIN
  {
    /* open file to append new information about trace-memory ... */
    fp=BESTX_FOPEN(file,"a");

    if (fp==NULL)
    {
      return BX_E_FILE_OPEN;
    }

    /* information text */
    BESTX_FPRINTF(  fp,
      "\n************************************************************************\n");
    BESTX_FPRINTF(  fp,"Testing Trace-memory\n");

    BESTX_FPRINTF(  fp,"Now starting test ...\n");
    BESTX_FPRINTF(  fp,"---------------------\n");

    /* check if E2929A or E2930A new RaM 26.02. */
    if (BestXHasFaust(handle)) /* E2930A */
    { 
      faust=1;
      memsize=6;  /* Memsize 192bit for E2930A */
      memdeep=22;
      maxreg=0xb; /*Set max column register (12 words per line) */
    }
    else /* E2929B */
    {
      faust=0;
      memsize=4;  /* Memsize 128bit for E2929B */
      memdeep=21;
      maxreg=0x7; /* Set max column register (8 words per line) */
    }

    /* check whether there is a deep trace board present */
    if (!faust)
    {
      BX_TRY(BestXStatusRead(handle, BX_STAT_PIGGYBACKID, &ReadVal));
      if (ReadVal != 1)
      { 
        /* no or wrong piggyback board detected -> return OK */
        BX_TRY_ERROR(BX_E_OK);           
      } /* end */
    }
    MemAddr = 0;
    Pattern = 0;

    /* initialize the trace memory for writing from dbi */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x21));

    /* set min max register */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_LATCH_READ_START_F:BX_REG_DT_LATCH_READ_START_M),sizeof(bx_int16),0));
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_LATCH_READ_STOP_F:BX_REG_DT_LATCH_READ_STOP_M),sizeof(bx_int16),maxreg));


    /* set the address counter */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M),sizeof(bx_int16),MemAddr));
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M),sizeof(bx_int16),MemAddr));

    /* set function enable bit and dbi write mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0xA));


    /* write one memory line: memsize * 32 Bit = 128/192 Bit*/
    for (i=0;i<memsize;i++)
    { /* write port is only 16 Bit */
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),Pattern>>16));
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),Pattern));
    } 

    /* turn off write mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x0));

    /* reset the address counter */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M),sizeof(bx_int16),MemAddr));
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M),sizeof(bx_int16),MemAddr));

    /* switch to read mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x6));

    for (i=0;i<memsize;i++)
    {
      BX_TRY(BestXDirectRegRead( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),&ReadValLo));
      BX_TRY(BestXDirectRegRead( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),&ReadValHi));

      ReadVal = (ReadValLo<<16) | ReadValHi;
      if (ReadVal != Pattern)
      {
        TestFail = BX_TRUE;
        BESTX_FPRINTF(  fp,"Error in trace memory data line test.\n");
        BESTX_FPRINTF(  fp,"Wrote Value %08X in column %04X and read back %08X\n", 
          Pattern, i, ReadVal);
      }
    } 

    /* turn off read mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x0));

    Pattern = 0xFFFFFFFF;

    /* set the address counter */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M),sizeof(bx_int16),MemAddr));
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M),sizeof(bx_int16),MemAddr));

    /* set function enable bit and dbi write mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0xA));


    /* write one memory line: memsize * 32 Bit = 128 Bit*/
    for (i=0;i<memsize;i++)
    {
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),Pattern>>16));
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),Pattern));
    } 

    /* turn off write mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x0));

    /* reset the address counter */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M),sizeof(bx_int16),MemAddr));
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M),sizeof(bx_int16),MemAddr));

    /* switch to read mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x6));

    for (i=0;i<memsize;i++)
    {
      BX_TRY(BestXDirectRegRead( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),&ReadValLo));
      BX_TRY(BestXDirectRegRead( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),&ReadValHi));

      ReadVal = (ReadValLo<<16) | ReadValHi;
      if (ReadVal != Pattern)
      {
        TestFail = BX_TRUE;
        BESTX_FPRINTF(  fp,"Error in trace memory data line test.\n");
        BESTX_FPRINTF(  fp,"Wrote Value %08X in column %04X and read back %08X\n", 
          Pattern, i, ReadVal);
      }
    } 
    /* turn off read mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x0));

    Pattern = 0xAAAAAAAA;

    /* set the address counter */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M),sizeof(bx_int16),MemAddr));
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M),sizeof(bx_int16),MemAddr));

    /* set function enable bit and dbi write mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0xA));


    /* write one memory line: memsize * 32 Bit = 128 Bit*/
    for (i=0;i<memsize;i++)
    {
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),Pattern>>16));
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),Pattern));
    } 

    /* turn off write mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x0));

    /* reset the address counter */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M),sizeof(bx_int16),MemAddr));
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M),sizeof(bx_int16),MemAddr));

    /* switch to read mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x6));

    for (i=0;i<memsize;i++)
    {
      BX_TRY(BestXDirectRegRead( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),&ReadValLo));
      BX_TRY(BestXDirectRegRead( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),&ReadValHi));

      ReadVal = (ReadValLo<<16) | ReadValHi;
      if (ReadVal != Pattern)
      {
        TestFail = BX_TRUE;
        BESTX_FPRINTF(  fp,"Error in trace memory data line test.\n");
        BESTX_FPRINTF(  fp,"Wrote Value %08X in column %04X and read back %08X\n", 
          Pattern, i, ReadVal);
      }
    } 
    /* turn off read mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x0));

    Pattern = 0x55555555;

    /* set the address counter */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M),sizeof(bx_int16),MemAddr));
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M),sizeof(bx_int16),MemAddr));

    /* set function enable bit and dbi write mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0xA));


    /* write one memory line memsize * 32 Bit = 128 Bit*/
    for (i=0;i<memsize;i++)
    {
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),Pattern>>16));
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),Pattern));
    } 

    /* turn off write mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x0));

    /* reset the address counter */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M),sizeof(bx_int16),MemAddr));
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M),sizeof(bx_int16),MemAddr));

    /* switch to read mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x6));

    for (i=0;i<memsize;i++)
    {
      BX_TRY(BestXDirectRegRead( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),&ReadValLo));
      BX_TRY(BestXDirectRegRead( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),&ReadValHi));

      ReadVal = (ReadValLo<<16) | ReadValHi;
      if (ReadVal != Pattern)
      {
        TestFail = BX_TRUE;
        BESTX_FPRINTF(  fp,"Error in trace memory data line test.\n");
        BESTX_FPRINTF(  fp,"Wrote Value %08X in column %04X and read back %08X\n", 
          Pattern, i, ReadVal);
      }
    } 
    /* turn off read mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x0));


    /* now test the address lines */


    Pattern = 0;
    MemAddr = 0;


    /* set the address counter */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M),sizeof(bx_int16),MemAddr>>16));
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M),sizeof(bx_int16),MemAddr));

    /* set function enable bit and dbi write mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0xA));


    /* write one memory line: memsize * 32 Bit = 128/192 Bit*/
    for (i=0;i<memsize;i++)
    { /* write port is only 16 Bit */
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),Pattern>>16));
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),Pattern));
    } 

    Pattern = 0x1;
    MemAddr = 0x1;



    for (j=0;j<memdeep;j++)
    {

      /* turn off write mode */
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x0));

      /* set the address counter */
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M),sizeof(bx_int16),MemAddr>>16));
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M),sizeof(bx_int16),MemAddr));

      /* set function enable bit and dbi write mode */
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0xA));


      /* write one memory line: memsize * 32 Bit = 128 Bit*/
      for (i=0;i<memsize;i++)
      {
        BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),Pattern>>16));
        BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),Pattern));
      }

      Pattern <<= 1;
      MemAddr <<= 1;

    }


    /* and now compare */

    Pattern = 0x0;
    MemAddr = 0x0;

    /* turn off write mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x0));

    /* set the address counter */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M),sizeof(bx_int16),MemAddr>>16));
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M),sizeof(bx_int16),MemAddr));

    /* switch to read mode */
    BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x6));

    for (i=0;i<memsize;i++)
    {
      BX_TRY(BestXDirectRegRead( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),&ReadValLo));
      BX_TRY(BestXDirectRegRead( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),&ReadValHi));

      ReadVal = ((ReadValLo<<16) | ReadValHi);
      if (ReadVal != Pattern)
      {
        TestFail = BX_TRUE;
        BESTX_FPRINTF(  fp,"Error in trace memory address line test.\n");
        BESTX_FPRINTF(  fp,"Wrote Value %08X in column %04X of address %08X and read back %08X\n", 
          Pattern, i, MemAddr, ReadVal);
      }
    } 

    Pattern = 0x1;
    MemAddr = 0x1;

    for (j=0;j<memdeep;j++)
    {
      /* turn off read mode */
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x0));

      /* set the address counter */
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_HI_F:BX_REG_DT_ADDR_HI_M),sizeof(bx_int16),MemAddr>>16));
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_ADDR_LO_F:BX_REG_DT_ADDR_LO_M),sizeof(bx_int16),MemAddr));

      /* switch to read mode */
      BX_TRY(BestXDirectRegWrite( handle,(BestXHasFaust(handle)?BX_REG_DT_CONTROL_F:BX_REG_DT_CONTROL_M),sizeof(bx_int16),0x6));

      for (i=0;i<memsize;i++)
      {
        BX_TRY(BestXDirectRegRead( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),&ReadValLo));
        BX_TRY(BestXDirectRegRead( handle,(BestXHasFaust(handle)?BX_REG_DT_DATAREAD_F:BX_REG_DT_DATAREAD_HI_M),sizeof(bx_int16),&ReadValHi));

        ReadVal = (ReadValLo<<16) | ReadValHi;
        if (ReadVal != Pattern)
        {
          TestFail = BX_TRUE;
          BESTX_FPRINTF(  fp,"Error in trace memory address line test.\n");
          BESTX_FPRINTF(  fp,"Wrote Value %08X in column %04X of address %08X and read back %08X\n", 
            Pattern, i, MemAddr, ReadVal);
        }
      } 
      Pattern <<= 1;
      MemAddr <<= 1;
    }

    BESTX_FPRINTF(  fp,"End of test on Trace-memory:\n");
    BESTX_FPRINTF(  fp,"\n************************************************************************\n");

    BESTX_FCLOSE(fp);

  } /* BX_TRY_BEGIN */

  if(BX_TRY_RET) BX_ERRETURN(BX_TRY_RET);
  if(!TestFail) BX_ERRETURN(BX_E_OK); else BX_ERRETURN(BX_E_SELFTEST_FAILED);


}
